; MUSE demonstration for CA65.
; by thefox//aspekt 2011

.include "read_joy.h"
.include "muse-flags.h"

.zeropage

; System that the ROM is running on (MUSE_Flags::NTSC_MODE or 0).
system:                     .res 1
; Current MUSE flags.
current_flags:              .res 1
; Current selected song.
song:                       .res 1
; Current selected sound effect.
sound_effect:               .res 1
; Current volume 0..255.
volume:                     .res 1
; Current controller state.
controller_state:           .res 1
; Previous controller state.
prev_controller_state:      .res 1
; Stores buttons which have been pressed down after previously
; being not pressed.
btn_down_controller_state:  .res 1
; Zeropage used by MUSE.
MUSE_ZEROPAGE:              .res 7

.bss

; RAM used by MUSE.
MUSE_RAM:                   .res 256

.enum Joy
    A_          = %1
    B           = %10
    SELECT      = %100
    START       = %1000
    UP          = %10000
    DOWN        = %100000
    LEFT        = %1000000
    RIGHT       = %10000000
.endenum

NUM_SONGS               = 4
NUM_SOUND_EFFECTS       = 4
FADE_SPEED              = 3
OAM                     = $200

; Increase a variable and clamp it to a maximum.
.macro incClamp addr, max
    .local no_overflow
    ldx addr
    inx
    cpx #max
    bne no_overflow
        ldx #max - 1
    no_overflow:
    stx addr
.endmacro

; Decrease a variable, clamp to 0.
.macro decClampToZero addr
    .local no_underflow
    ldx addr
    dex
    bpl no_underflow
        ldx #0
    no_underflow:
    stx addr
.endmacro

; Upload a byte as a hexadecimal number to PPU.
.macro uploadHex ppu_addr, what
    lda #>ppu_addr
    sta $2006
    lda #<ppu_addr
    sta $2006
    lda what
    lsr
    lsr
    lsr
    lsr
    tax
    lda hex_lut, x
    sta $2007
    lda what
    and #$F
    tax
    lda hex_lut, x
    sta $2007
.endmacro

.code

.proc init
    sei
    cld
    
    lda #0
    sta $2000
    sta $2001
    
    ; Wait for the PPU to stabilize.
    jsr pollVblank
    jsr pollVblank
    
    ; Clear all RAM (except $100-2FF).
    lda #0
    tax
    :
        sta $0,x
        sta $300,x
        sta $400,x
        sta $500,x
        sta $600,x
        sta $700,x
        inx
    bne :-
    
    ; Clear shadow OAM to $FF.
    lda #$FF
    :
        sta $200, x
        inx
    bne :-
    
    ; Disable APU frame IRQ.
    lda #$40
    sta $4017

    rts
.endproc

.proc detectSystem
    jsr pollVblank
    
    ; This loop takes a bit over 30k cycles.
    ldy #24
    :
        ldx #0
        :
            inx
        bne :-
        dey
    bne :--
    
    lda #1
    bit $2002
    bmi is_ntsc
    ; Is PAL.
    lda #0
is_ntsc:
    rts
.endproc

.export reset
.proc reset
    ldx #$ff
    txs
    
    jsr init
    jsr detectSystem
    cmp #0
    beq not_ntsc
        lda #MUSE_Flags::NTSC_MODE
    not_ntsc:
    sta system
    
    ; Initialize MUSE.
    lda #<snd_data
    ldx #>snd_data
    jsr MUSE_init
    
    ; Set a flag so that global volume doesn't affect sound effects. Also
    ; set PAUSE so that we can safely call MUSE_update even when a song isn't
    ; loaded.
    lda #MUSE_Flags::PAUSE|MUSE_Flags::GLOBAL_VOL_BEFORE_SFX
    ora system
    sta current_flags
    jsr MUSE_setFlags
    
    ; Initialize global volume.
    lda #$F0
    sta volume
    
    ; Setup sprite 0.
    ; Y.
    lda #8 * $13 - 1
    sta OAM + 0
    ; Tile.
    lda #$78
    sta OAM + 1
    ; Attributes.
    lda #%00100000
    sta OAM + 2
    ; X.
    lda #8 * $D
    sta OAM + 3
    
    jsr uploadPalette
    jsr uploadNametable
    jsr uploadSprites

    ; Wait for vblank and set the scroll (also enables rendering and the NMI).
    jsr pollVblank
    jsr setScroll
    
    jmp *
.endproc

.proc pollVblank
    bit $2002
    :
        bit $2002
    bpl :-
    rts
.endproc

.proc uploadPalette
    lda #$3F
    sta $2006
    lda #0
    sta $2006

    ldx #0
    :
        lda palette, x
        sta $2007
        inx
        cpx #16
    bne :-
    
    rts
.endproc

.proc uploadNametable
    lda #$20
    sta $2006
    lda #0
    sta $2006

    ldx #0
    .repeat 4, i
        :
            lda nametable + 256 * i, x
            sta $2007
            inx
        bne :-
    .endrepeat
    
    rts
.endproc

.proc updateControllerState
    lda controller_state
    sta prev_controller_state
    jsr read_joy
    sta controller_state
    
    lda prev_controller_state
    eor #$FF
    and controller_state
    sta btn_down_controller_state
    
    rts
.endproc

.proc uploadSprites
    lda #0
    sta $2003
    lda #2
    sta $4014
    rts
.endproc

.proc uploadNametableChanges
    ; Song number.
    uploadHex $2169, song
    ; Sound effect number.
    uploadHex $2189, sound_effect
    
    rts
    
hex_lut:
    .repeat 10, i
        .byte '0' + i
    .endrepeat
    .repeat 6, i
        .byte 'A' + i
    .endrepeat
.endproc

.proc setScroll
    ; Set the scroll.
    lda #0
    sta $2005
    sta $2005
    ; Enable NMI and set the MSB of scroll.
    lda #$80
    sta $2000
    ; Enable rendering.
    lda #%00011000
    sta $2001
    rts
.endproc

.proc waitForSprite0Hit
    ; Wait for the flag to be cleared (end of vblank).
    :
        bit $2002
    bvs :-
    
    ; Wait for the flag to be set.
    :
        bit $2002
    bvc :-
    
    rts
.endproc

.proc nmi
    jsr uploadSprites
    jsr uploadNametableChanges
    jsr setScroll

    ; Wait for sprite 0 hit.
    jsr waitForSprite0Hit
    ; Delay a couple of scanlines.
    ldx #0
    :
        nop
        nop
        dex
    bne :-
    
    ; Set intensity.
    lda #%11111000
    sta $2001

    jsr MUSE_update

    ; Disable intensity.
    lda #%00011000
    sta $2001
    
    jsr updateControllerState
    
    ; UP/DOWN changes the song.
    lda btn_down_controller_state
    and #Joy::UP
    beq :+
        incClamp song, NUM_SONGS
    :
    lda btn_down_controller_state
    and #Joy::DOWN
    beq :+
        decClampToZero song
    :
    
    ; LEFT/RIGHT changes the sound effect.
    lda btn_down_controller_state
    and #Joy::RIGHT
    beq :+
        incClamp sound_effect, NUM_SOUND_EFFECTS
    :
    lda btn_down_controller_state
    and #Joy::LEFT
    beq :+
        decClampToZero sound_effect
    :
    
    ; A/B starts the music/sound effect.
    lda btn_down_controller_state
    and #Joy::A_
    beq :+
        lda song
        jsr MUSE_startMusic
        ; Make sure pause is off.
        lda current_flags
        and #<~MUSE_Flags::PAUSE_MUSIC
        sta current_flags
        jsr MUSE_setFlags
        ; Set the volume to 0 to start fading up.
        lda #0
        sta volume
    :
    lda btn_down_controller_state
    and #Joy::B
    beq :+
        ldx sound_effect
        ; If it's the 4th effect, stop it if it's playing, because it loops.
        cpx #3
        bne not_fourth
            jsr MUSE_isSfxPlaying
            beq not_playing
                ; If already playing, stop it.
                jsr MUSE_stopSfx
                jmp :+
            not_playing:
        not_fourth:
        ; Play on the same channel as the sfx number.
        txa
        jsr MUSE_startSfx
        ; Make sure pause is off.
        lda current_flags
        and #<~MUSE_Flags::PAUSE_SFX
        sta current_flags
        jsr MUSE_setFlags
    :
    
    ; START pauses/resumes the music and sound effects.
    lda btn_down_controller_state
    and #Joy::START
    beq :+
        lda current_flags
        eor #MUSE_Flags::PAUSE
        sta current_flags
        jsr MUSE_setFlags
    :
    
    ; Fade up the volume.
    ldx volume
    inx
    inx
    ; If already at maximum, don't save.
    beq :+
        stx volume
    :
    ; Divide by 16 to get from 0..255 down to 0..15.
    lda volume
    lsr
    lsr
    lsr
    lsr
    jsr MUSE_setVolume

    rti
.endproc

.proc irq
    rti
.endproc

; MUSE library, must be aligned to a 256 byte page.
    .align 256
    .include "muse-ca65.h"

.rodata

palette:
    .incbin "gfx/muse.pal"

nametable:
    .incbin "gfx/muse.nam"

; General data.
    .include "snd-data/snd-data.s"
; Song data.
    .include "snd-data/snd-data-jabadaa.s"
    .include "snd-data/snd-data-ripulimiehenkosto.s"
    .include "snd-data/snd-data-cheetah.s"
    .include "snd-data/snd-data-cursedjungle.s"
; Sound effect data.
    .include "snd-data/snd-data-sfx-ding.s"
    .include "snd-data/snd-data-sfx-metal.s"
    .include "snd-data/snd-data-sfx-tri.s"
    .include "snd-data/snd-data-sfx-paskasireeni.s"

.segment "DPCM"
; DPCM data (must be at $C000-FFFF and aligned to 64 bytes).
    .include "snd-data/snd-data-dpcm.s"
    
.segment "CHR"
    .incbin "gfx/muse.chr"

.segment "VECTORS"
    .word nmi
    .word reset
    .word irq
