// PC2NES
// A tool for transferring iNES ROMs to PowerPak over a serial cable.
// by thefox//aspekt 2010-2011
// Uses NRPC library by blargg

#include "nrpc/nrpc.h"
#include <boost/asio.hpp>
#include <iostream>
#include <string>
#include <fstream>
#include <sstream>
#include <iomanip>

// Set to 0 to transfer all of the data at 57600 bps. Otherwise send
// a small amount of data at 57600 bps and the rest at 115200 bps.
#define SEND_115200             1
#define NRPC_DIR                "nrpc-1.2a1"

using namespace std;
using namespace boost;

typedef unsigned char uchar;

// If str is not NULL, prints it and exits
static void error(const char* str);

// PowerPak registers.
const nrpc_addr_t prg_bank     = 0x4200;
const nrpc_addr_t chr_bank     = 0x4201;
const nrpc_addr_t mirroring    = 0x4202;
const nrpc_addr_t gamegenie    = 0x4204;
const nrpc_addr_t boot_en      = 0x4207;
const nrpc_addr_t has_battery  = 0x4208;
const nrpc_addr_t fpga_program = 0x5800;
const nrpc_addr_t fpga_data    = 0x5C00;

const int MAX_GAME_GENIE_CODES = 5;

#pragma pack(push, 1)
struct iNESHeader {
    uchar sig[4]; // "NES\x1A"
    uchar num_16k_prg_banks;
    uchar num_8k_chr_banks;
    uchar flags6;
    uchar flags7;
    uchar num_8k_wram_banks;
    uchar flags9;
    uchar flags10;
    uchar dummy[5];
};
#pragma pack(pop)

void change_bgcolor(nrpc_t *nes, int bgcolor)
{
    // Set the background color.
    nrpc_write_byte(nes, 0x2006, 0x3F);
    nrpc_write_byte(nes, 0x2006, 0x00);
    nrpc_write_byte(nes, 0x2007, bgcolor);

    // Reset PPU address away from palette area so that background
    // color is displayed.
    nrpc_write_byte(nes, 0x2006, 0x00);
    nrpc_write_byte(nes, 0x2006, 0x00);
}

string parse_valarg_old(string option)
{
    string::size_type equ_pos = option.find('=');
    if(equ_pos == string::npos) {
        return "";
    } else {
        return option.substr(equ_pos + 1);
    }
}

bool parse_valarg(const string& option, string& result)
{
    string::size_type equ_pos = option.find('=');
    if(equ_pos == string::npos) {
        return false;
    } else {
        result = option.substr(equ_pos + 1);
        return true;
    }
}

struct GameGenieCode
{
    GameGenieCode() : addr( -1 ), compare( -1 ), data( -1 ) {}
    int addr;
    int compare;
    int data;
};

int translate_gg_char( char ch )
{
    switch( ch )
    {
        case 'A': case 'a': return 0;
        case 'P': case 'p': return 1;
        case 'Z': case 'z': return 2;
        case 'L': case 'l': return 3;
        case 'G': case 'g': return 4;
        case 'I': case 'i': return 5;
        case 'T': case 't': return 6;
        case 'Y': case 'y': return 7;
        case 'E': case 'e': return 8;
        case 'O': case 'o': return 9;
        case 'X': case 'x': return 10;
        case 'U': case 'u': return 11;
        case 'K': case 'k': return 12;
        case 'S': case 's': return 13;
        case 'V': case 'v': return 14;
        case 'N': case 'n': return 15;
    }

    return -1;
}

bool decode_gamegenie( const string& code, GameGenieCode& result )
{
    // source: http://tuxnes.sourceforge.net/gamegenie.html

    if( code.size() != 6 && code.size() != 8 )
    {
        return false;
    }

    int n[ 8 ];
    for( int i = 0; i < code.size(); ++i )
    {
        n[ i ] = translate_gg_char( code[ i ] );
        if( n[ i ] == -1 )
        {
            return false;
        }
    }

    result = GameGenieCode();

    result.addr = 0x8000 + 
        ((n[ 3 ] & 7) << 12)
        | ((n[ 5 ] & 7) << 8) | ((n[ 4 ] & 8) << 8)
        | ((n[ 2 ] & 7) << 4) | ((n[ 1 ] & 8) << 4)
        |  (n[ 4 ] & 7)       |  (n[ 3 ] & 8);

    if( code.size() == 6 )
    {
        result.data =
             ((n[ 1 ] & 7) << 4) | ((n[ 0 ] & 8) << 4)
            | (n[ 0 ] & 7)       |  (n[ 5 ] & 8);
    }
    else
    {
        result.data =
             ((n[ 1 ] & 7) << 4) | ((n[ 0 ] & 8) << 4)
            | (n[ 0 ] & 7)       |  (n[ 7 ] & 8);
        result.compare =
             ((n[ 7 ] & 7) << 4) | ((n[ 6 ] & 8) << 4)
            | (n[ 6 ] & 7)       |  (n[ 5 ] & 8);
    }

    return true;
}

// Delay for "amount" milliseconds.
void delay( int amount )
{
    boost::asio::io_service io;

    boost::asio::deadline_timer timer( io );
    timer.expires_from_now( boost::posix_time::milliseconds( amount ) );
    timer.wait();
}

int main(int argc, char **argv)
{
    if(argc < 2)
    {
show_usage:
        cerr << "usage: " << argv[0] << " [options] infile" << endl << endl;
        cerr << "    -ntsc            use NTSC timing" << endl;
        cerr << "    -pal             use PAL timing" << endl;
        cerr << "    -wram=[file]     WRAM contents" << endl;
        cerr << "    -gg=[code]       Game Genie code" << endl;
        cerr << "    -port=[port]     send data to a serial port" << endl;
        cerr << "    -out=[file]      save data to a file" << endl;
        return EXIT_SUCCESS;
    }

    int last_arg = argc - 1;
    int nes_system = -1;
    string serial_port;
    string outfilename;
    string wramfilename;
    vector< GameGenieCode > game_genie_codes;

    // Parse options.
    for(int i = 1; i < last_arg; ++i) {
        string option = argv[i];
        if(option == "-pal") {
            nes_system = nrpc_pal;
        } else if(option == "-ntsc") {
            nes_system = nrpc_ntsc;
        } else if(option.substr(0, 6) == "-port=") {
            if( !parse_valarg(option, serial_port) )
            {
                goto invalid_option;
            }
        } else if(option.substr(0, 5) == "-out=") {
            if( !parse_valarg(option, outfilename) )
            {
                goto invalid_option;
            }
        } else if(option.substr(0, 6) == "-wram=") {
            if( !parse_valarg(option, wramfilename) )
            {
                goto invalid_option;
            }
        } else if(option.substr(0, 4) == "-gg=") {
            string gg_code;
            if( !parse_valarg(option, gg_code) )
            {
                goto invalid_option;
            }

            GameGenieCode code;
            if( decode_gamegenie( gg_code, code ) )
            {
                if( game_genie_codes.size() < MAX_GAME_GENIE_CODES )
                {
                    game_genie_codes.push_back( code );
                }
                else
                {
                    cerr << "warning: game genie code '" << gg_code <<
                        "' ignored" << endl;
                }
            }
            else
            {
                cerr << "warning: invalid game genie code '" << gg_code <<
                    "' ignored" << endl;
            }
            
        } else {
invalid_option:
            cerr << "invalid option: " << option << endl;
            goto show_usage;
        }
    }

    if(nes_system < 0) {
        cerr << "error: need to specify either PAL or NTSC (see usage)" << endl;
        return EXIT_FAILURE;
    }

    ifstream inesfs(argv[last_arg], ios::binary);
    if(!inesfs.is_open()) {
        cerr << "error: couldn't open infile" << endl;
        return EXIT_FAILURE;
    }

    ifstream wramfs;
    if( !wramfilename.empty() )
    {
        wramfs.open( wramfilename, ios::binary );
        if( !wramfs.is_open() )
        {
            cerr << "error: couldn't open wram file" << endl;
            return EXIT_FAILURE;
        }
    }

    iNESHeader header;
    inesfs.read((char *)&header, sizeof header);

    if(memcmp(header.sig, "NES\x1A", sizeof header.sig) != 0) {
        cerr << "error: infile is not a valid iNES ROM" << endl;
        return EXIT_FAILURE;
    }

    int mapper = (header.flags7 & 0xF0) | (header.flags6 >> 4);

    int battery_backed_wram = (header.flags6 & 2) ? 1 : 0;

    ostringstream mapper_file;
    mapper_file << "mappers/MAP" << hex << uppercase << setfill('0') << setw(2) << mapper << ".MAP";
    ifstream mapperfs(mapper_file.str(), ios::binary);
    if(!mapperfs.is_open()) {
        cerr << "error: couldn't open mapper file '" << mapper_file.str() << '\'' << endl;
        return EXIT_FAILURE;
    }
    
    const int xfer_speed = SEND_115200 ? nrpc_115200 : nrpc_57600;
    nrpc_t* nes = nrpc_new(nes_system | xfer_speed);
    if(!nes)
    {
        cerr << "error: out of memory" << endl;
        return EXIT_FAILURE;
    }

#ifdef _DEBUG
    error(nrpc_load_library(nes, NRPC_DIR "/nrpc.lib"));
    error(nrpc_load_library(nes, NRPC_DIR "/thefox-routines/foxnrpc.lib"));
#else
    error(nrpc_load_library(nes, "nrpc/nrpc.lib"));
    error(nrpc_load_library(nes, "nrpc/foxnrpc.lib"));
#endif

    // Some padding to allow checking for controller while waiting for
    // transfer.
    nrpc_zero_pad( nes, 512 );

    int num_8k_prg_banks = header.num_16k_prg_banks * 2;
    int num_8k_chr_banks = header.num_8k_chr_banks;

    // Generic buffer.
    static unsigned char buf[0x10000];

    // disable rendering
    nrpc_write_byte(nes, 0x2000, 0x00);
    nrpc_write_byte(nes, 0x2001, 0x00);

    // Clear 4 rows.
    nrpc_fill_ppu( nes, 0x21A0, ' ', 4 * 32);

    // Display message.
    nrpc_write_ppu(nes, 0x21A7, (const uchar *)"Receiving PRG...    ", 20);

    // nrpc_write_ppu() has disabled rendering, so enable it again.
    nrpc_write_byte(nes, 0x2006, 0x00);
    nrpc_write_byte(nes, 0x2006, 0x00);
    nrpc_write_byte(nes, 0x2000, 0x00);
    nrpc_write_byte(nes, 0x2001, 0x0A);

    for(int i = 0; i < num_8k_prg_banks; ++i)
    {
        inesfs.read((char *)buf, 0x2000);
        nrpc_write_byte(nes, prg_bank, i);
        nrpc_write_mem(nes, 0x6000, buf, 0x2000);
    }

    if( wramfs.is_open() )
    {
        // Copy message to PPU memory and enable rendering.
        nrpc_write_ppu(nes, 0x21A7, (const uchar *)"Receiving WRAM...   ", 20);
        nrpc_write_byte(nes, 0x2006, 0x00);
        nrpc_write_byte(nes, 0x2006, 0x00);
        nrpc_write_byte(nes, 0x2000, 0x00);
        nrpc_write_byte(nes, 0x2001, 0x0A);

        int wram_bank = 0;
        while( wramfs.read((char *)buf, 0x2000) )
        {
            // Top bit indicates that WRAM is mapped to 6000-7FFF instead
            // of prg.
            nrpc_write_byte(nes, prg_bank, wram_bank | 0x80);
            nrpc_write_mem(nes, 0x6000, buf, 0x2000);
            ++wram_bank;
        }
    }

    // Disable rendering and change the background color to red to
    // indicate CHR upload.
    nrpc_write_byte(nes, 0x2000, 0x00);
    nrpc_write_byte(nes, 0x2001, 0x00);
    change_bgcolor(nes, 0x06);

    for(int i = 0; i < num_8k_chr_banks; ++i)
    {
        inesfs.read((char *)buf, 0x2000);
        nrpc_write_byte(nes, chr_bank, i);
        nrpc_write_ppu(nes, 0, buf, 0x2000);
    }

    // Change to blue to indicate mapper upload.
    change_bgcolor(nes, 0x01);

    // Form the gamegenie data at the beginning of the buffer.
    const int GAMEGENIE_DATA_SIZE = 30;
    memset( buf, 0, GAMEGENIE_DATA_SIZE );
    {
        unsigned char* p = buf;
        for( auto it = game_genie_codes.begin(); it !=
            game_genie_codes.end(); ++it )
        {
            *p++ = it->data; // newbyte
            *p++ = it->compare == -1 ? 0 : 0xFF; // enable
            *p++ = it->compare == -1 ? 0 : it->compare; // compare
            *p++ = it->addr & 0xFF; // addr low
            *p++ = it->addr >> 8; // addr hight
        }
    }

    // Here's the "WRAM reg".
    // PowerPak uses this at boot time to determine whether the game uses
    // SRAM so it can display the menu.
    buf[ 25 ] = battery_backed_wram;
    buf[ 28 ] = 0x08;
    buf[ 29 ] = 0x42; // Register at 4208.
    
    const int MAPPER_SIZE = 42166;
    // Skip the mapper loader.
    mapperfs.seekg(0x400);
    // Read the mapper in the buffer right after the Game Genie data.
    mapperfs.read((char *)buf + GAMEGENIE_DATA_SIZE, MAPPER_SIZE);

    // Start FPGA programming.
    nrpc_write_byte(nes, fpga_program, 0xFF);
    // Call a custom routine to program the FPGA.
    nrpc_call(nes, "write_fpga_config", 0, 0, 0, 0);
    // Send the data to write_fpga_config.
    nrpc_send_block(nes, buf, GAMEGENIE_DATA_SIZE + MAPPER_SIZE);

    // write_fpga_config needs some cycles to write the Game Genie settings.
    nrpc_delay_cycles(nes, 500);

    // Change background color to black to indicate everything is ready.
    change_bgcolor(nes, 0xF);
    
    // Write hardwired mirroring.
    nrpc_write_byte(nes, mirroring, header.flags6 & 1);

    // Write prg and chr size masks (for CHR-RAM use $81).
    nrpc_write_byte(nes, prg_bank, num_8k_prg_banks - 1);
    if(num_8k_chr_banks)
    {
        nrpc_write_byte(nes, chr_bank, num_8k_chr_banks * 2 - 1);
    }
    else
    {
        nrpc_write_byte(nes, chr_bank, 0x81);
    }

    // Disable boot rom and jump to the reset vector.
    // Note that some mappers (my Save State Mappers in particular) enable
    // controller bus capacitance emulation when boot_en is 0. The transfer
    // code relies on the high 3 bits being always set, so no further
    // transfers can be made after setting boot_en to 0 when using those
    // mappers.
    unsigned char code [] = {
        0xA9,0x00,                          // lda #00
        0x8D,boot_en & 0xFF,boot_en >> 8,   // sta boot_en
        0x6C,0xFC,0xFF,                     // jmp ($FFFC)
    };
    nrpc_call_code( nes, 0x200, code, sizeof code, 0x200, 0, 0,
            0, 0, 0, 0 );
    
    if(!serial_port.empty())
    {
        int data_size;
        const uchar *data = nrpc_recording(nes, &data_size);
        if(data)
        {
            using namespace boost::asio;

            try
            {
                io_service io;
                asio::serial_port port(io, serial_port);

                // send the first X bytes at 57600 baud rate
                port.set_option(serial_port_base::baud_rate(57600));
                port.set_option(serial_port_base::character_size(8));
                port.set_option(serial_port_base::flow_control(serial_port_base::flow_control::none));
                port.set_option(serial_port_base::parity(serial_port_base::parity::none));
                port.set_option(serial_port_base::stop_bits(serial_port_base::stop_bits::one));
                
                {
                    const int transfer_size = nrpc_57600_count(nes);
                    cout << "sending " << (transfer_size / 1024.f) << " KB to serial port " << serial_port << " at 57600 bps" << endl;
                    size_t num_bytes_written = asio::write(port, buffer(data, transfer_size));
                    data += transfer_size;
                    data_size -= transfer_size;
                    assert(data_size >= 0);
                }

                delay( 100 );

                // rest of the data at 115200 bps
                port.set_option(serial_port_base::baud_rate(115200));
                port.set_option(serial_port_base::stop_bits(serial_port_base::stop_bits::two));

                cout << "sending " << (data_size / 1024.f) << " KB to serial port " << serial_port << " at 115200 bps" << endl;

                const int BLOCK_SIZE = 512;
                int data_transfered = 0;
                int old_completed = -1;
                while(data_transfered < data_size) {
                    const int data_left = data_size - data_transfered;
                    const int transfer_size = data_left < BLOCK_SIZE ? data_left : BLOCK_SIZE;
                    size_t num_bytes_written = asio::write(port, buffer(data + data_transfered, transfer_size));
                    int completed = data_transfered * 100 / data_size;
                    if(completed != old_completed) {
                        cout << "\r  " << setw(3) << completed << " % completed";
                        old_completed = completed;
                    }
                    data_transfered += transfer_size;
                }
                if(old_completed != 100) {
                    cout << "\r  100 % completed";
                }
                cout << endl;
            } catch(const std::exception& e) {
                cerr << "error: Boost.Asio exception: " << e.what() << endl;
                return EXIT_FAILURE;
            }
        }
    }

    if(!outfilename.empty())
    {
        error(nrpc_save_split_recording(nes, (outfilename + '1').c_str(), (outfilename + '2').c_str()));
    }
    
    nrpc_delete(nes);
    
    return EXIT_SUCCESS;
}

void error(const char* str)
{
    if (str != NULL)
    {
        cerr <<  "nrpc error: " << str << endl;
        exit(EXIT_FAILURE);
    }
}
