/* Produces demo.bin that when sent to NES, displays
HELLO WORLD on screen in changing colors. Requires CHR RAM */

#include "nrpc/nrpc.h"

/* If str is not NULL, prints it and exits */
static void error( const char* str );

extern const unsigned char alphabet [26 * 0x10];

int main( void )
{
	int flags = nrpc_pal | nrpc_115200;
	
	/* Create new recorder */
	nrpc_t* nes = nrpc_new( flags );
	if ( nes == 0 )
		error( "out of memory" );
	
	/* Load standard routine library */
	error( nrpc_load_library( nes, "nrpc.lib" ) );
	
	/* Load our routine (comment out to use local version below) */
	error( nrpc_load_library( nes, "cycle_colors.bin" ) );
	
	/* Blacken palette */
	nrpc_fill_ppu( nes, 0x3F00, 0x0F, 0x20 );

	/* Load uppercase letters into CHR RAM */
	nrpc_fill_ppu( nes, 0x0000, 0x00, 0x2000 );
	nrpc_write_ppu( nes, 'A' * 0x10, alphabet, 26 * 0x10 );
	
	/* Load nametable */
	nrpc_fill_ppu( nes, 0x2000, 0, 0x400 );
	{
		unsigned char str [] = "HELLO WORLD";
		nrpc_write_ppu( nes, 0x210A, str, 11 );
	}
	
	/* If our color cycle routine is in library, call it there */
	if ( nrpc_routine_exists( nes, "cycle_colors" ) )
	{
		nrpc_call( nes, "cycle_colors", 0, 0, 0, 0 );
	}
	else
	{
		/* Otherwise run it from array in this code */
		unsigned char code [] = {
		                    /* .org $200 */
			0x2C,0x02,0x20, /* bit $2002 */
			0x10,0xFB,      /* bpl $0000 */
			0xA2,0x3F,      /* ldx #$3f */
			0x8E,0x06,0x20, /* stx $2006 */
			0xA0,0x03,      /* ldy #$00 */
			0x8C,0x06,0x20, /* sty $2006 */
			0x8D,0x07,0x20, /* sta $2007 */
			0xA0,0x00,      /* ldy #$00 */
			0x8C,0x00,0x20, /* sty $2000 */
			0x8C,0x05,0x20, /* sty $2005 */
			0x8C,0x05,0x20, /* sty $2005 */
			0xA2,0x0A,      /* ldx #$0a */
			0x8E,0x01,0x20, /* stx $2001 */
			0x69,0x40,      /* adc #$40 */
			0x4C,0x00,0x02, /* jmp $0200 */
		};
		
		nrpc_call_code( nes, 0x200, code, sizeof code, 0x200, 0, 0,
				0, 0, 0, 0 );
	}

	/* Save session to file(s), ready to be sent to NES via serial.
	Second file is empty if 57600 serial speed is used, so that it
	can always be sent at 115200, if desired. */
	error( nrpc_save_split_recording( nes, "demo.bin", "demo.bin2" ) );
	
	/* Free memory */
	nrpc_delete( nes );
	
	return 0;
}

const unsigned char alphabet [26 * 0x10] = {
	0x38,0x6C,0xC6,0xC6,0xFE,0xC6,0xC6,0x00,0x38,0x6C,0xC6,0xC6,0xFE,0xC6,0xC6,0x00,
	0xFC,0xC6,0xC6,0xFC,0xC6,0xC6,0xFC,0x00,0xFC,0xC6,0xC6,0xFC,0xC6,0xC6,0xFC,0x00,
	0x3C,0x66,0xC0,0xC0,0xC0,0x66,0x3C,0x00,0x3C,0x66,0xC0,0xC0,0xC0,0x66,0x3C,0x00,
	0xF8,0xCC,0xC6,0xC6,0xC6,0xCC,0xF8,0x00,0xF8,0xCC,0xC6,0xC6,0xC6,0xCC,0xF8,0x00,
	0xFE,0xC0,0xC0,0xFC,0xC0,0xC0,0xFE,0x00,0xFE,0xC0,0xC0,0xFC,0xC0,0xC0,0xFE,0x00,
	0xFE,0xC0,0xC0,0xFC,0xC0,0xC0,0xC0,0x00,0xFE,0xC0,0xC0,0xFC,0xC0,0xC0,0xC0,0x00,
	0x3E,0x60,0xC0,0xCE,0xC6,0x66,0x3E,0x00,0x3E,0x60,0xC0,0xCE,0xC6,0x66,0x3E,0x00,
	0xC6,0xC6,0xC6,0xFE,0xC6,0xC6,0xC6,0x00,0xC6,0xC6,0xC6,0xFE,0xC6,0xC6,0xC6,0x00,
	0x3C,0x18,0x18,0x18,0x18,0x18,0x3C,0x00,0x3C,0x18,0x18,0x18,0x18,0x18,0x3C,0x00,
	0x1E,0x06,0x06,0x06,0xC6,0xC6,0x7C,0x00,0x1E,0x06,0x06,0x06,0xC6,0xC6,0x7C,0x00,
	0xC6,0xCC,0xD8,0xF0,0xD8,0xCC,0xC6,0x00,0xC6,0xCC,0xD8,0xF0,0xD8,0xCC,0xC6,0x00,
	0x60,0x60,0x60,0x60,0x60,0x60,0x7E,0x00,0x60,0x60,0x60,0x60,0x60,0x60,0x7E,0x00,
	0xC6,0xEE,0xFE,0xFE,0xD6,0xC6,0xC6,0x00,0xC6,0xEE,0xFE,0xFE,0xD6,0xC6,0xC6,0x00,
	0xC6,0xE6,0xF6,0xFE,0xDE,0xCE,0xC6,0x00,0xC6,0xE6,0xF6,0xFE,0xDE,0xCE,0xC6,0x00,
	0x7C,0xC6,0xC6,0xC6,0xC6,0xC6,0x7C,0x00,0x7C,0xC6,0xC6,0xC6,0xC6,0xC6,0x7C,0x00,
	0xFC,0xC6,0xC6,0xFC,0xC0,0xC0,0xC0,0x00,0xFC,0xC6,0xC6,0xFC,0xC0,0xC0,0xC0,0x00,
	0x7C,0xC6,0xC6,0xC6,0xDA,0xCC,0x76,0x00,0x7C,0xC6,0xC6,0xC6,0xDA,0xCC,0x76,0x00,
	0xFC,0xC6,0xC6,0xFC,0xD8,0xCC,0xC6,0x00,0xFC,0xC6,0xC6,0xFC,0xD8,0xCC,0xC6,0x00,
	0x7C,0xC6,0xC0,0x7C,0x06,0xC6,0x7C,0x00,0x7C,0xC6,0xC0,0x7C,0x06,0xC6,0x7C,0x00,
	0xFC,0x30,0x30,0x30,0x30,0x30,0x30,0x00,0xFC,0x30,0x30,0x30,0x30,0x30,0x30,0x00,
	0xC6,0xC6,0xC6,0xC6,0xC6,0xC6,0x7C,0x00,0xC6,0xC6,0xC6,0xC6,0xC6,0xC6,0x7C,0x00,
	0xC6,0xC6,0xC6,0xEE,0x7C,0x38,0x10,0x00,0xC6,0xC6,0xC6,0xEE,0x7C,0x38,0x10,0x00,
	0xC6,0xC6,0xD6,0xFE,0xFE,0xEE,0xC6,0x00,0xC6,0xC6,0xD6,0xFE,0xFE,0xEE,0xC6,0x00,
	0xC6,0xEE,0x7C,0x38,0x7C,0xEE,0xC6,0x00,0xC6,0xEE,0x7C,0x38,0x7C,0xEE,0xC6,0x00,
	0x66,0x66,0x66,0x3C,0x18,0x18,0x18,0x00,0x66,0x66,0x66,0x3C,0x18,0x18,0x18,0x00,
	0xFE,0x0E,0x1C,0x38,0x70,0xE0,0xFE,0x00,0xFE,0x0E,0x1C,0x38,0x70,0xE0,0xFE,0x00,
};

#include <stdlib.h>
#include <stdio.h>

void error( const char* str )
{
	if ( str != NULL )
	{
		fprintf( stderr, "Error: %s\n", str );
		exit( EXIT_FAILURE );
	}
}
