#include "Nrpc_Core.h"

#include <assert.h>
#include <stdlib.h>
#include <string.h>

/* Copyright (C) 2010 Shay Green. This module is free software; you
can redistribute it and/or modify it under the terms of the GNU Lesser
General Public License as published by the Free Software Foundation; either
version 2.1 of the License, or (at your option) any later version. This
module is distributed in the hope that it will be useful, but WITHOUT ANY
WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public License for more
details. You should have received a copy of the GNU Lesser General Public
License along with this module; if not, write to the Free Software Foundation,
Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA */

typedef unsigned char byte;

int const delay_value = 0xFF;

Nrpc_Core::Nrpc_Core()
{
	recording_       = NULL;
	recording_size_  = 0;
	dummy            = 0;
	count_57600      = 0;
	serial_115200    = false;
	lengthen_delays_ = false;
}

void Nrpc_Core::clear_recording()
{
	free( recording_ );
	recording_      = NULL;
	recording_size_ = 0;
	count_57600     = 0;
}

static int rounded( int n )
{
	assert( n >= 0 );
	int const chunk = 24 * 1024;
	return (n + chunk-1)/chunk * chunk;
}

// Makes space for appending size bytes and returns pointer, or NULL if out of memory
Nrpc_Core::byte* Nrpc_Core::expand( int size )
{
	if ( recording_size_ < 0 )
		return NULL;
	
	int current = rounded( recording_size_ );
	int needed  = rounded( recording_size_ + size );
	if ( needed > current )
	{
		void* p = realloc( recording_, needed );
		if ( p == NULL )
		{
			clear_recording();
			recording_size_ = -1;
			return NULL;
		}
		
		recording_ = (byte*) p;
	}
	
	if ( !serial_115200 )
		count_57600 += size;
	
	recording_size_ += size;
	return recording_ + recording_size_ - size;
}

const byte* Nrpc_Core::recording( int* size_out )
{
	*size_out = 0;
	
	if ( recording_size_ < 0 )
		return NULL;
	
	// avoid returning NULL, even if no data
	*size_out = recording_size_;
	return (recording_ == NULL ? &dummy : recording_);
}

void Nrpc_Core::send_raw( const byte in [], int size )
{
	byte* out = expand( size );
	if ( out == NULL )
		return;
	
	memcpy( out, in, size );
}

void Nrpc_Core::send( const byte in [], int size )
{
	byte* out = expand( size );
	if ( out == NULL )
		return;
	
	for ( int i = 0; i < size; i++ )
	{
		int flipped = in [i];
		
		// Reverse bit order
		if ( serial_115200 )
		{
			for ( int n = 0; n < 8; n++ )
				flipped = (flipped << 1) | (in [i] >> n & 1);
		}
		
		out [i] = flipped;
	}
}

void Nrpc_Core::put( int n )
{
	byte b = n;
	send( &b, 1 );
}

void Nrpc_Core::delay_bytes( int count )
{
	if ( lengthen_delays_ )
		count = count * 2 + 10;
	
	const byte b [1] = { delay_value };
	while ( count-- )
		send_raw( b, sizeof b );
}

static int calc_byte_crc( int b, int crc )
{
	crc ^= b;
	crc <<= 1;
	crc += crc >> 8 & 1;
	crc += 0x99;
	crc &= 0xFF;
	
	return crc;
}

int Nrpc_Core::calc_crc( const byte in [], int count, int crc )
{
	for ( int i = 0; i < count; i++ )
		crc = calc_byte_crc( in [i], crc );
	
	return crc;
}

int Nrpc_Core::calc_crc_inv( const byte in [], int count, int crc )
{
	for ( int i = count; i--; )
	{
		crc += 0x100 - 0x99;
		crc = (crc << 7) | (crc >> 1 & 0x7F);
		crc &= 0xFF;
		crc ^= in [i];
	}
	return crc;
}

int const header_size = 4 + 0x10;

void Nrpc_Core::make_header( byte out [header_size],
		int in_crc, int in_size, int addr, int exec,
		int arg0, int arg1, int arg2, int arg3 )
{
	const byte zero = 0;
	in_crc = calc_crc_inv( &zero, 1, in_crc );

	int neg_size = -in_size;
	int adj_addr = addr - (neg_size & 0xFF);
	
	byte header [header_size] = {
		0xD4,0x5B,0xCC,0x2B,
		arg0, arg0 >> 8,
		arg1, arg1 >> 8,
		arg2, arg2 >> 8,
		arg3, arg3 >> 8,
		exec, exec >> 8,
		adj_addr, adj_addr >> 8,
		in_crc,
		0,
		neg_size, neg_size >> 8
	};
	
	int const beg = 0x04;
	int const mid = 0x11;
	
	int crc = calc_byte_crc( 0, 0 );
	crc = Nrpc_Core::calc_crc( &header [beg], mid - beg, crc );
	crc ^= Nrpc_Core::calc_crc_inv( header + mid, header_size - mid - 1,
			header [header_size - 1] );
	header [mid] = crc;
	
	memcpy( out, header, header_size );
}

void Nrpc_Core::call_code_( int dest, const byte in [], int size, int exec,
			const byte* extra, int extra_size, int arg0, int arg1, int arg2, int arg3  )
{
	if ( exec == -1 )
		exec = dest;
	
	assert( 0 <= dest && dest < mem_size );
	assert( 0 <= extra_size );
	assert( 0 <= size );
	assert( dest + size + extra_size <= mem_size );
	assert( 0 <= exec && exec < mem_size );
	
	const byte zero = 0;
	if ( size <= 0 )
	{
		in   = &zero;
		size = 1;
		dest = 0x3F;
	}
	
	int crc = calc_byte_crc( 0, 0 );
	crc = calc_crc(    in,       size, crc );
	crc = calc_crc( extra, extra_size, crc );
	
	byte header [header_size];
	make_header( header, crc, size + extra_size, dest,
			exec, arg0, arg1, arg2, arg3 );
	
	send( header, sizeof header );
	send( in, size );
	send( extra, extra_size );
}

