// Makes recording NES remote procedure calls

#ifndef BLARGG_NRPC_CORE_H
#define BLARGG_NRPC_CORE_H

#include "nrpc.h"
#include <stdlib.h>

class Nrpc_Core {
public:
	
	typedef unsigned char byte;
	
// Data upload to NES

	// Sends individual bytes without any additional protocol
	void send( const byte in [], int size );
	
	// Sends bytes without any encoding. Won't be readable by the loader's
	// standard read routine.
	void send_raw( const byte in [], int size );

	// Allows additional cycles for execution by inserting padding bytes
	// in serial stream. Without this, bytes would be lost.
	void delay_bytes( int bytes );

// Access to recorded data

	// Returns pointer and size of data to send to NES. Does not clear it.
	// NULL if out of memory.
	const byte* recording( int* size_out );
	
	int recording_57600_count() const   { return count_57600; }
	
	// Clears recording
	void clear_recording();
	
// Debugging

	void debug_lengthen_delays()    { lengthen_delays_ = true; }
	
private:
	// noncopyable
	Nrpc_Core( const Nrpc_Core& );
	Nrpc_Core& operator = ( const Nrpc_Core& );
	
public:
	Nrpc_Core();
	~Nrpc_Core() { clear_recording(); }
	
	// Make operator new() fail via NULL rather than exception
	
	// throw spec mandatory in ISO C++ if NULL can be returned
	#if __cplusplus >= 199711 || __GNUC__ >= 3 || _MSC_VER >= 1300
		#define NES_THROWS_NOTHING throw ()
	#else
		#define NES_THROWS_NOTHING
	#endif
	
	void* operator new ( size_t s ) NES_THROWS_NOTHING { return malloc( s ); }
	void operator delete( void* p ) NES_THROWS_NOTHING { free( p ); }
	
	
protected:
	enum { mem_size = 0x10000 };
	
	static int calc_crc( const byte in [], int count, int crc );
	static int calc_crc_inv( const byte in [], int count, int crc );
	
	void begin_115200() { serial_115200 = true; }
	
	void put( int );
	void call_code_( int dest, const byte* in, int size, int exec,
			const byte* extra, int extra_size, int arg0, int arg1, int arg2, int arg3  );

private:
	byte* recording_;
	int  recording_size_;
	int  count_57600;
	bool serial_115200;
	bool lengthen_delays_;
	byte dummy;
	
	byte* expand( int size );
	static void make_header( byte out [],
			int in_crc, int in_size, int addr, int exec,
			int arg0, int arg1, int arg2, int arg3 );
};

#endif
