#include "Nrpc_Library.h"

#include <assert.h>
#include <stddef.h>
#include <string.h>
#include <stdlib.h>
#include <stdio.h>

/* Copyright (C) 2010 Shay Green. This module is free software; you
can redistribute it and/or modify it under the terms of the GNU Lesser
General Public License as published by the Free Software Foundation; either
version 2.1 of the License, or (at your option) any later version. This
module is distributed in the hope that it will be useful, but WITHOUT ANY
WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public License for more
details. You should have received a copy of the GNU Lesser General Public
License along with this module; if not, write to the Free Software Foundation,
Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA */

char const Nrpc_Library::signature [signature_size] = { 'S','U','B','R' };

Nrpc_Library::Nrpc_Library()
{
	file      = NULL;
	file_size = 0;
}

Nrpc_Library::~Nrpc_Library()
{
	free( file );
}

inline void Nrpc_Library::begin()
{
	iter = 0;
}

static const char* append_file( unsigned char** data, int* size, const char path [] )
{
	long file_size;
	unsigned char* moved;
	FILE* in;
	
	in = fopen( path, "rb" );
	if ( in == NULL )
		return "couldn't open file";
	
	fseek( in, 0, SEEK_END );
	file_size = ftell( in );
	fseek( in, 0, SEEK_SET );
	if ( file_size < 0 || ferror( in ) )
		return "couldn't get file size";
	
	moved = (unsigned char*) realloc( *data, *size + file_size + 1 );
	if ( moved == NULL )
		return "out of memory";
	
	*data = moved;
	
	if ( (long) fread( moved + *size, 1, file_size, in ) < file_size )
		return "couldn't read file";
	
	fclose( in );
	
	*size += file_size;
	
	return NULL;
}

Nrpc_Library::err_t Nrpc_Library::load_library_( int new_size )
{
	int old_size = file_size;
	file_size = new_size;
	
	// Try updated library
	err_t err = corrupt();
	if ( err )
	{
		file_size = old_size;
		return err;
	}
	
	begin();
	
	return NULL;
}

Nrpc_Library::err_t Nrpc_Library::load_library( const char path [] )
{
	int new_size = file_size;
	err_t err = append_file( &file, &new_size, path );
	if ( !err )
		err = load_library_( new_size );
	
	return err;
}

Nrpc_Library::err_t Nrpc_Library::load_library( const unsigned char* in, int size )
{
	int new_size = file_size + size;
	void* moved = realloc( file, new_size + 1 );
	if ( moved == NULL )
		return "out of memory";
	
	file = (unsigned char*) moved;
	memcpy( &file [file_size], in, size );
	
	return load_library_( new_size );
}

int Nrpc_Library::header( field_t field ) const
{
	assert( iter + field_size <= file_size );
	
	int offset = iter + signature_size + field*field_size;
	return file [offset] + file [offset + 1] * 0x100;
}

Nrpc_Library::err_t Nrpc_Library::corrupt()
{
	iter = 0;
	while ( iter < file_size )
	{
		// Careful of reading header before we know it's not truncated
		
		if ( iter + signature_size <= file_size &&
				memcmp( &file [iter], signature, signature_size ) )
		{
			if ( iter == 0 )
				return "not a routine library file";
			
			break;
		}
		
		if ( iter + min_header_size > file_size )
			break;
		
		int header_size = header( f_header_size );
		if ( iter + header_size >= file_size )
			break;
		
		if ( header_size < min_header_size )
			break;
		
		int size = header( f_size );
		if ( iter + size > file_size )
			break;
		
		int name_end = header_size;
		for ( ; name_end < size; name_end++ )
			if ( file [iter + name_end] == 0 )
				break;
		
		if ( name_end >= size )
			break;
		
		if ( header( f_data ) + header( f_data_size ) > size )
			break;
		
		iter += header( f_size );
	}
	
	if ( iter == file_size )
		return NULL;
	
	return "corrupt routine library";
}

Nrpc_Library::routine_t const* Nrpc_Library::next()
{
	assert( 0 <= iter && iter <= file_size );
	
	if ( iter >= file_size )
		return NULL;
	
	result.name  = (const char*) &file [iter + header( f_header_size )];
	result.code  = &file [iter + header( f_data )];
	result.flags = header( f_flags );
	result.info  = header( f_info );
	result.size  = header( f_data_size );
	result.addr  = header( f_addr );
	result.exec  = header( f_exec );
	
	iter += header( f_size );
	
	return &result;
}

Nrpc_Library::routine_t const* Nrpc_Library::find( const char name [], int flags )
{
	routine_t best = { NULL, NULL, 0, 0, 0, 0, 0 };
	
	begin();
	routine_t const* r;
	while ( (r = next()) != NULL )
	{
		if ( !strcmp( r->name, name ) )
		{
			// Flags matched so far
			int best_match = flags & best.flags;
			
			// Flags current one matches
			int curr_match = flags & r->flags;
			
			// Must match at least the ones matched so far
			if ( (curr_match & best_match) == best_match )
				best = *r;
		}
	}
	
	if ( best.name == NULL )
		return NULL;
	
	result = best;
	return &result;
}
