// Library of named NES routines

#ifndef BLARGG_NRPC_LIBRARY_H
#define BLARGG_NRPC_LIBRARY_H

class Nrpc_Library {
public:
	typedef const char* err_t;
	
	// Loads library from file. Same-named routines
	// are replaced with those in the most-recently loaded.
	err_t load_library( const char path [] );
	
	// Same as load_library(), but copies from memory.
	err_t load_library( const unsigned char*, int size );
	
	struct routine_t {
		const char* name;
		unsigned char const* code;
		int flags;
		int info;
		int size;
		int addr;
		int exec;
	};
	
	// Returns routine with specified name, or NULL if not found.
	// If more than one routine of same name, returns LAST one.
	routine_t const* find( const char name [], int flags = 0 );
	
	// Begins iterating over all routines in library, including
	// duplicates.
	void begin();

	// Pointer to information for next routine, or NULL if at end.
	routine_t const* next();

public:
	Nrpc_Library();
	~Nrpc_Library();

private:
	routine_t result;
	unsigned char* file;
	int file_size;
	int iter;
	
	enum field_t {
		f_size,
		f_header_size,
		f_flags,
		f_info,
		f_data,
		f_data_size,
		f_addr,
		f_exec
	};
	
	enum { field_size = 2 };
	enum { signature_size = 4 };
	enum { min_header_size = signature_size + (f_exec + 1) * field_size };
	
	static char const signature [signature_size];
	
	err_t corrupt();
	int header( field_t ) const;
	err_t load_library_( int new_size );
};

#endif
