#include "Nrpc_Loader.h"

#include <assert.h>
#include <stdlib.h>
#include <string.h>

/* Copyright (C) 2010 Shay Green. This module is free software; you
can redistribute it and/or modify it under the terms of the GNU Lesser
General Public License as published by the Free Software Foundation; either
version 2.1 of the License, or (at your option) any later version. This
module is distributed in the hope that it will be useful, but WITHOUT ANY
WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public License for more
details. You should have received a copy of the GNU Lesser General Public
License along with this module; if not, write to the Free Software Foundation,
Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA */

typedef unsigned char byte;

int const cycles_per_byte_57600 = 280;
int const bytes_per_sec_57600   = 5760;

Nrpc_Loader::Nrpc_Loader()
{
	loader_          = NULL;
	loader_size_     = 0;
	loader_sent      = false;
	ack_count        = 0;
	bootloader_vers  = 0;
	cycles_per_byte_ = cycles_per_byte_57600;
	bytes_per_sec_   = bytes_per_sec_57600;
	serial_115200    = false;
}

void Nrpc_Loader::enable_115200()
{
	cycles_per_byte_ = cycles_per_byte_57600 / 2;
	bytes_per_sec_   = bytes_per_sec_57600 * 2;
	serial_115200    = true;
}

void Nrpc_Loader::delay_cycles( int cycles )
{
	delay_bytes( cycles / cycles_per_byte_ + 1 );
}

void Nrpc_Loader::delay_msec( int msec )
{
	int inflated_bytes_per_sec = bytes_per_sec_ * 33 / 32;
	delay_bytes( (msec * inflated_bytes_per_sec + 1000-1) / 1000 );
}

void Nrpc_Loader::expect_ack()
{
	ack_count++;
	delay_bytes( 1 );
}

void Nrpc_Loader::call_code( addr_t dest, const byte in [], int size, addr_t exec,
			const byte* extra, int extra_size, int arg0, int arg1, int arg2, int arg3 )
{
	if ( !loader_sent )
		resend_loader();
	
	delay_bytes( 1 );
	call_code_( dest, in, size, exec, extra, extra_size, arg0, arg1, arg2, arg3 );
	expect_ack();
}

void Nrpc_Loader::send_block( const byte in [], int size )
{
	const byte zero [1] = { 0 };
	
	int crc = 0;
	if ( size > 0 )
	{
		crc = in [size - 1];
		crc = calc_crc_inv( in, size - 1, crc );
		crc = calc_crc_inv( zero, 1, crc );
	}
	crc ^= calc_crc( zero, sizeof zero, 0 );
	
	delay_bytes( 1 );
	put( 'B' );
	put( crc );
	send( in, size );
	
	expect_ack();
}

void Nrpc_Loader::write_mem( addr_t dest, const byte in [], int size )
{
	assert( 0 <= dest && dest <= mem_size );
	assert( 0 <= size && size <= mem_size - dest );
	
	if ( size <= 0 )
		return;
	
	// Let loader do it for us, then execute arg2, which is 0x60, RTS
	call_code( dest, in, size, 0x34, NULL, 0, 0x6060, 0x6060, 0x6060, 0x6060 );
}

void Nrpc_Loader::ping()
{
	const byte rts = 0x60;
	write_mem( 0x30, &rts, 1 );
}

enum { boot_block_size = 256 };

/* Converts 256-byte block of user code (beginning at offset 7)
into 256-byte program block ready to send to boot loader */
static void make_boot_block_v1( byte block [boot_block_size], int old )
{
	enum { boot_block_offset = 7 };
	const byte header [2] [boot_block_offset] = {
		{ 0xDC, 0x4B, 0xD2, 0x3B, 0, 0, 0 },
		{ 0xE2, 0x5D, 0xCC, 0x75, 0, 0, 0 },
	};
	long crc;
	int i, n;
	
	/* Write header to beginning of block */
	for ( i = 0; i < boot_block_offset; i++ )
		block [i] = header [old] [i];
	
	/* Calculate 16-bit CRC that will cancel out in the end */
	crc = 0;
	for ( i = 255; i >= 5; i-- )
	{
		for ( n = 0; n < 8; n++ )
			crc = (crc >> 1) ^ ((crc & 1) * 0x8810);
		
		crc = crc ^ (block [i] << 8);
	}
	block [5] = crc >> 8 & 0xFF;
	block [6] = crc      & 0xFF;
	
	/* Calculate 8-bit checksum AFTER 16-bit CRC */
	n = -block [0];
	for ( i = 0; i < boot_block_size; i++ )
		n = n - block [i];
	block [4] = n;
	
	if ( old )
	{
		/* Reverse bit order and complement bits */
		for ( i = 0; i < boot_block_size; i++ )
		{
			int flipped = 0;
			for ( n = 0; n < 8; n++ )
				flipped = (flipped << 1) | (block [i] >> n & 1);
			
			block [i] = flipped ^ 0xFF;
		}
	}
}

enum { boot_block_offset = 4 };

/* Converts 256-byte block of user code (beginning at offset 4)
into 256-byte program block ready to send to boot loader */
static void make_boot_block( unsigned char block [boot_block_size] )
{
	int i, crc = 0;
	for ( i = boot_block_size - 1; i >= boot_block_offset - 1; i-- )
	{
		crc += 0x100 - 0x99;
		crc = (crc << 7 & 0x80) | (crc >> 1 & 0x7F);
		crc ^= block [i];
	}
	block [0]  = 0xDC;
	block [1]  = 0x4B;
	block [2]  = 0xD2;
	block [3] ^= crc ^ 0xCB;
}

static void make_boot( byte out [boot_block_size], const byte in [], int size, int vers )
{
	assert( size > 0 && size <= boot_block_size );
	
	memset( out, 0, boot_block_size );
	memcpy( out, in, size );
	
	switch ( vers )
	{
	case 1:
		make_boot_block_v1( out, true );
		break;
	
	case 2:
		make_boot_block_v1( out, false );
		break;
		
	default:
		make_boot_block( out );
		break;
	}
}

void Nrpc_Loader::send_boot_( const byte in [], int size, int exec, int reg_x, int reg_y )
{
	byte block [boot_block_size];
	make_boot( block, in, size, bootloader_vers );
	
	if ( bootloader_vers == 1 )
		exec = -1;
	
	if ( exec < 0 )
	{
		send_raw( block, boot_block_size );
	}
	else
	{
		static const byte reload [] = {
			0xA4,0x46,      // ldy $46
			0xA2,0x00,      // ldx #$00
			0xBD,0x14,0x02, // lda $0214,x
			0x95,0x00,      // sta $00,x
			0xE8,           // inx
			0xD0,0xF8,      // bne $0004
			0x98,           // tya
			0xA2,reg_x,     // ldx #reg_x
			0xA0,reg_y,     // ldy #reg_y
			0x4C,exec,exec>>8,  // jmp exec
		};
		
		// Prepend load command so it works regardless of whether bootloader or
		// nrpc loader is running
		delay_bytes( 1 );
		put( 0 ); // tell loader to ignore any garbage before
		call_code_( 0x200, reload, sizeof reload, -1,
				block, boot_block_size, 0, 0, 0, 0 );
	}
	
	bootloader_vers = 0;
}

Nrpc_Loader::err_t Nrpc_Loader::set_loader( const byte* loader, int size, int exec )
{
	(void) exec;
	
	if ( loader != NULL )
		if ( size < 0x100 || size > max_loader_size )
			return "loader is wrong size";
	
	loader_      = loader;
	loader_size_ = size;
	
	return NULL;
}

void Nrpc_Loader::resend_loader()
{
	loader_sent = true;
	
	if ( loader_ == NULL )
		return;
	
	if ( serial_115200 )
	{
		delay_bytes( 1 );
		send_boot_( loader_, boot_size, -1 );
		delay_bytes( 32 );
		begin_115200();
		delay_bytes( 31 );
		
		send_boot_( loader_, boot_size, 0x0009, 0, 1 );
		delay_bytes( 63 );
	}
	else
	{
		send_boot( loader_, boot_size );
		delay_bytes( 63 );
	}
	
	// Remaining code, in reverse order, so PHA can be used to write it
	int remain_size = loader_size_ - boot_size;
	byte remain [256];
	for ( int i = 0; i < remain_size; i++ )
		remain [i] = loader_ [boot_size + remain_size - 1 - i];
	
	int crc = calc_crc_inv( remain, remain_size - 1,
			remain [remain_size - 1] );
	crc = calc_crc_inv( (const byte*) "\0", 1, crc );
	const byte zero [1] = { 0 };
	crc ^= calc_crc( zero, sizeof zero, 0 );
	
	put( 0x5A );
	put( crc );
	send( remain, remain_size );
	delay_bytes( 1 );
	
	expect_ack();
}
