// Makes recording NES remote procedure calls

#ifndef BLARGG_NRPC_LOADER_H
#define BLARGG_NRPC_LOADER_H

#include "Nrpc_Core.h"

class Nrpc_Loader : public Nrpc_Core {
public:

	// NULL on success, otherwise pointer to error string
	typedef nrpc_err_t err_t;
	
	// Address in NES memory
	typedef int addr_t;
	
// Serial rate
	
	void enable_115200();
	
	int bytes_per_sec() const   { return bytes_per_sec_; }

// Boot

	// Sends as boot program, loaded into zero-page. First boot_addr
	// bytes are not loaded. Execution begins there.
	enum { boot_addr = 7 };
	enum { boot_size = 256 };
	void send_boot( const byte in [], int size )    { send_boot_( in, size ); }
	
	// deprecated
	void use_old_bootloader( int vers ) { bootloader_vers = vers; }
	void scramble_bootprog()    { bootloader_vers = 1; }
	
// Loader

	// Sets loader for resend_loader() to use. Returns error if loader is
	// of unsuitable size.
	enum { max_loader_size = 0x200 - 3 };
	err_t set_loader( const byte* loader, int size, int exec );
	
	// Resends loader, clearing fatal error if one occurred
	void resend_loader();
	
// Remote procedure calls

	// Uploads size byte of code from in to dest memory, then executes it at exec
	// (dest if exec isn't specified). Appends extra_size bytes to routine's code.
	// Passes optional 16-bit arguments to routine.
	void call_code( addr_t dest, const byte in [], int size, addr_t exec = -1,
			const byte* extra = NULL, int extra_size = 0,
			int arg0 = 0, int arg1 = 0, int arg2 = 0, int arg3 = 0 );
	
	// Sends synchronized, checksummed block of data to executing routine
	void send_block( const byte in [], int size );
	
	// Writes data to NES RAM
	void write_mem( addr_t dest, const byte in [], int size );
	
	void delay_cycles( int cycles );
	void delay_msec( int msec );

// Acknowledgement
	
	// Generates ack
	void ping();
	
	// Increments acknowledgements count and delays by a byte
	void expect_ack();
	
	// Returns acknowledgements count and clears it
	int clear_acks()                { int n = ack_count; ack_count = 0; return n; }

public:
	Nrpc_Loader();
	
private:
	byte const* loader_;
	int loader_size_;
	bool loader_sent;
	int  bootloader_vers;
	
	int ack_count;
	
	int cycles_per_byte_;
	int bytes_per_sec_;
	bool serial_115200;
	
	void send_boot_( const byte in [], int size,
			int exec = boot_addr, int reg_x = 0, int reg_y = 0 );
};

#endif
