#include "Nrpc_Recorder.h"

#include <assert.h>
#include <string.h>
#include <stdio.h>

/* Copyright (C) 2010 Shay Green. This module is free software; you
can redistribute it and/or modify it under the terms of the GNU Lesser
General Public License as published by the Free Software Foundation; either
version 2.1 of the License, or (at your option) any later version. This
module is distributed in the hope that it will be useful, but WITHOUT ANY
WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public License for more
details. You should have received a copy of the GNU Lesser General Public
License along with this module; if not, write to the Free Software Foundation,
Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA */

int const ppu_size = 0x4000;

nrpc_t::nrpc_t()
{
	missing_routines_ = false;
	routine_flags     = 0x50;
	debug             = false;
}

bool Nrpc_Recorder::missing_routines()
{
	bool result = missing_routines_;
	missing_routines_ = false;
	return result;
}

Nrpc_Library::routine_t const* Nrpc_Recorder::find( const char name [] )
{
	return lib.find( name, routine_flags );
}

Nrpc_Recorder::err_t Nrpc_Recorder::finish_load()
{
	Nrpc_Library::routine_t const* r = find( "nrpc_boot" );
	if ( r == NULL )
		return "loader is missing";
	
	// Ignore loader if zero-sized (presumably one isn't wanted in that case)
	if ( r->size != 0 )
	{
		if ( r->addr != 0 || r->size < 0x100 || r->size > max_loader_size )
			return "nrpc_boot routine has wrong size/address";
		
		err_t err = set_loader( r->code, r->size, r->exec );
		if ( err )
			return err;
	}
	
	return NULL;
}

Nrpc_Recorder::err_t Nrpc_Recorder::enable_pal_serial()
{
	routine_flags = (routine_flags & ~0x40) | 0x80;
	return finish_load();
}

Nrpc_Recorder::err_t Nrpc_Recorder::enable_115200_serial()
{
	routine_flags = (routine_flags & ~0x10) | 0x20;
	enable_115200();
	return finish_load();
}

Nrpc_Recorder::err_t Nrpc_Recorder::load_library( const char path [] )
{
	err_t err = lib.load_library( path );
	if ( !err )
		err = finish_load();
	
	return err;
}

Nrpc_Recorder::err_t Nrpc_Recorder::load_library( const byte* in, int size )
{
	err_t err = lib.load_library( in, size );
	if ( !err )
		err = finish_load();
	
	return err;
}

bool Nrpc_Recorder::routine_exists( const char name [] )
{
	return find( name ) != NULL;
}

void Nrpc_Recorder::call_extra( const char name [], const byte* extra, int extra_size,
		int arg0, int arg1, int arg2, int arg3 )
{
	Nrpc_Library::routine_t const* r = find( name );
	if ( r == NULL )
	{
		missing_routines_ = true;
		if ( debug )
			fprintf( stderr, "NES %s() not found\n", name );
		return;
	}

	if ( debug )
		fprintf( stderr, "NES %s( %04X, %04X, %04X, %04X )\n",
				name, (unsigned) arg0 & 0xFFFF, (unsigned) arg1 & 0xFFFF,
				(unsigned) arg2 & 0xFFFF, (unsigned) arg3 & 0xFFFF );
	
	call_code( r->addr, r->code, r->size, r->exec, extra, extra_size,
			arg0, arg1, arg2, arg3 );
}

void Nrpc_Recorder::call( const char name [], int arg0, int arg1, int arg2, int arg3 )
{
	call_extra( name, NULL, 0, arg0, arg1, arg2, arg3 );
}

void Nrpc_Recorder::jsr( addr_t addr )
{
	assert( 0 <= addr && addr < mem_size );
	
	call_code( 0, NULL, 0, addr );
}

void Nrpc_Recorder::write_byte( addr_t dest, int data )
{
	byte b = data;
	base::write_mem( dest, &b, 1 );
}

static int adj_addr( int addr, int size )
{
	return addr - (-size & 0xFFu);
}

void Nrpc_Recorder::fill_mem( addr_t addr, int fill, int size )
{
	assert( 0 <= addr && addr <= mem_size );
	assert( 0 <= size && size <= mem_size - addr );
	
	if ( size <= 0 )
		return;
	
	call( "fill_mem", adj_addr( addr, size ), -size, fill );
	delay_cycles( size * 10 + (size / 256) * 4 );
}

void Nrpc_Recorder::write_ppu_unopt( addr_t addr, const byte* in, int size )
{
	assert( 0 <= addr && addr <= ppu_size );
	assert( 0 <= size && size <= ppu_size - addr );
	
	if ( size <= 0 )
		return;
	
	call( "write_ppu", addr, -size );
	send_block( in, size );
}

void Nrpc_Recorder::fill_ppu( addr_t addr, int fill, int size )
{
	assert( 0 <= addr && addr <= ppu_size );
	assert( 0 <= size && size <= ppu_size - addr );
	
	if ( size <= 0 )
		return;
	
	call( "fill_ppu", addr, -size, fill );
	delay_cycles( size * 9 + (size / 256) * 4 );
}

void Nrpc_Recorder::write_mmc1( addr_t addr, int value, bool reset )
{
	assert( 0x8000 <= addr && addr <= 0xFFFF );
	
	int bits = 5;
	value &= 0x1F;
	
	if ( reset )
	{
		value = (value << 1) | 0x80;
		bits = 6;
	}
	
	call( "write_mmc1", addr, value, bits );
}

void Nrpc_Recorder::read_mem( addr_t addr, int size )
{
	assert( 0 <= addr && addr <= mem_size );
	assert( 0 <= size && size <= mem_size - addr );
	
	call( "read_mem", adj_addr( addr, size ), -size );
	delay_bytes( size * 5 / 4 );
}

void Nrpc_Recorder::read_ppu( addr_t addr, int size )
{
	assert( 0 <= addr && addr <= ppu_size );
	assert( 0 <= size && size <= ppu_size - addr );
	
	call( "read_ppu", addr, -size );
	delay_bytes( size * 5 / 4 );
}

void Nrpc_Recorder::read_crc()
{
	call( "read_crc" );
	delay_bytes( 3 );
}

int Nrpc_Recorder::calc_crc( const byte* in, int count, int crc )
{
	return base::calc_crc( in, count, crc );
}

int const threshold = 50;

// Breaks array into runs of the same byte, and runs that need to be copied.
// Returns length of next run to be copied, or negative length if it's a run.
// Only runs of threshold or more are recognized.
static int next_run( const unsigned char* in, int size )
{
	int i = 0;
	while ( i < size )
	{
		int start = i;
		while ( i < size && in [i] == in [start] )
			i++;
		
		if ( i - start >= threshold )
			return start ? start : -i;
	}
	
	return size;
}

void Nrpc_Recorder::write_mem( addr_t addr, const byte* in, int size )
{
	assert( 0 <= addr && addr <= mem_size );
	assert( 0 <= size && size <= mem_size - addr );
	
	// Avoid optimizing writes to lower part of memory
	int noopt = 0x230 - addr;
	if ( noopt > 0 )
	{
		// Do optimized FIRST, since it could overwrite earlier memory
		int remain = size - noopt;
		if ( remain > 0 )
			write_mem( addr + noopt, in + noopt, remain );
		
		base::write_mem( addr, in, noopt );
		return;
	}
	
	while ( size > 0 )
	{
		int n = next_run( in, size );
		if ( n < 0 )
			fill_mem( addr, *in, (n = -n) );
		else
			base::write_mem( addr, in, n );
		
		addr += n;
		in   += n;
		size -= n;
	}
}

void Nrpc_Recorder::write_ppu( addr_t addr, const byte* in, int size )
{
	assert( 0 <= addr && addr <= ppu_size );
	assert( 0 <= size && size <= ppu_size - addr );
	
	while ( size > 0 )
	{
		int n = next_run( in, size );
		if ( n < 0 )
			fill_ppu( addr, *in, (n = -n) );
		else
			write_ppu_unopt( addr, in, n );
		
		addr += n;
		in   += n;
		size -= n;
	}
}
