// Core remote procedures and call-by-name of any other routines in library

#ifndef BLARGG_NRPC_RECORDER_H
#define BLARGG_NRPC_RECORDER_H

#include "Nrpc_Loader.h"
#include "Nrpc_Library.h"

/* allows clean integration with C interface */
typedef struct nrpc_t Nrpc_Recorder;

struct nrpc_t : public Nrpc_Loader {
	typedef Nrpc_Loader base;
public:
// Configuration
	
	// Enables PAL serial. Should be called before calling any routines.
	err_t enable_pal_serial();
	
	// Enables 115200 serial. Should be called before calling any routines.
	err_t enable_115200_serial();
	
// Routine library

	// Loads library from file. Same-named routines
	// are replaced with those in the most-recently loaded.
	err_t load_library( const char path [] );
	
	// Same as load_library(), but copies from memory.
	err_t load_library( const byte*, int size );
	
	// True if named routine exists
	bool routine_exists( const char name [] ); // TODO: make const

// Named routine invocation

	// Calls named routine with arguments
	void call( const char name [],
			int arg0 = 0, int arg1 = 0, int arg2 = 0, int arg3 = 0 );
	
	// Same as call(), but appends extra data to routine
	void call_extra( const char name [], const byte* extra, int extra_size,
			int arg0 = 0, int arg1 = 0, int arg2 = 0, int arg3 = 0 );
	
	// True if any new attempts were made to call routines not in library.
	// Clears flag on return.
	bool missing_routines();

// Routines for execution

	// Calls routine already in memory at given address
	void jsr( addr_t );
	
// Routines for writing memory

	// Writes anywhere in CPU address space
	void write_byte( addr_t dest, int data );
	void write_mem( addr_t dest, const byte* in, int size );
	void fill_mem( addr_t dest, int fill, int size );
	
	// Writes anywhere in PPU address space
	void write_ppu( addr_t dest, const byte* in, int size );
	void fill_ppu( addr_t dest, int fill, int size );
	
	// Writes to MMC1. If reset is true, resets shift register first.
	void write_mmc1( addr_t addr, int value, bool reset = false );
	
// Routines for reading memory
	
	// Data is sent back to PC, without any protocol. Use read_crc() to
	// verify integrity.
	
	// Reads from CPU/PPU
	void read_mem( addr_t, int size );
	void read_ppu( addr_t, int size );
	
	// Reads 2-byte CRC of all sent data before this, then resets it.
	// To verify CRC, treat it as two bytes appended to the data to calc_crc().
	void read_crc();
	
	// Calculates CRC of received data. After all data and 2-byte CRC have
	// been passed, returns correct_crc if correct. Pass 0 for old_crc
	// of first calc_crc(), otherwise previous value.
	enum { correct_crc = nrpc_correct_crc };
	static int calc_crc( const byte* in, int count, int old_crc = 0 );
	
// Debugging
	
	void enable_debugging( bool b = true ) { debug = b; }
	
public:
	nrpc_t();
	Nrpc_Library& library()             { return lib; }
	
private:
	Nrpc_Library lib;
	int routine_flags;
	bool missing_routines_;
	bool debug;
	
	err_t finish_load();
	Nrpc_Library::routine_t const* find( const char name [] );
	
	void write_mem_unopt( addr_t dest, const byte* in, int size );
	void write_ppu_unopt( addr_t dest, const byte* in, int size );
};

#endif
