#include "nrpc.h"
#include "Nrpc_Recorder.h"
#include <string.h>
#include <stdio.h>

/* Copyright (C) 2010 Shay Green. This module is free software; you
can redistribute it and/or modify it under the terms of the GNU Lesser
General Public License as published by the Free Software Foundation; either
version 2.1 of the License, or (at your option) any later version. This
module is distributed in the hope that it will be useful, but WITHOUT ANY
WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public License for more
details. You should have received a copy of the GNU Lesser General Public
License along with this module; if not, write to the Free Software Foundation,
Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA */

typedef unsigned char byte;

void nrpc_delete( nrpc_t* o )                                           { delete o; }
nrpc_err_t nrpc_load_library( nrpc_t* o, const char path [] )           { return o->load_library( path ); }

void nrpc_write_byte( nrpc_t* o, nrpc_addr_t a, int v )                 { o->write_byte( a, v ); }
void nrpc_write_mem ( nrpc_t* o, nrpc_addr_t a, const byte* p, int s )  { o->write_mem ( a, p, s ); }
void nrpc_fill_mem  ( nrpc_t* o, nrpc_addr_t a, int fill, int s )       { o->fill_mem  ( a, fill, s ); }
void nrpc_write_ppu ( nrpc_t* o, nrpc_addr_t a, const byte* p, int s )  { o->write_ppu ( a, p, s ); }
void nrpc_fill_ppu  ( nrpc_t* o, nrpc_addr_t a, int fill, int s )       { o->fill_ppu  ( a, fill, s ); }
void nrpc_write_mmc1( nrpc_t* o, nrpc_addr_t a, int v, nrpc_bool_t r )  { o->write_mmc1( a, v, r ); }

void nrpc_read_mem( nrpc_t* o, nrpc_addr_t a, int s )                   { o->read_mem( a, s ); }
void nrpc_read_ppu( nrpc_t* o, nrpc_addr_t a, int s )                   { o->read_ppu( a, s ); }
void nrpc_read_crc( nrpc_t* o )                                         { o->read_crc(); }
int nrpc_calc_crc( nrpc_t* o, const byte* p, int s, int c )             { return o->calc_crc( p, s, c ); }

int nrpc_routine_exists( nrpc_t* o, const char name [] )                { return o->routine_exists( name ); }
int nrpc_missing_routines( nrpc_t* o )                                  { return o->missing_routines(); }
void nrpc_jsr( nrpc_t* o, nrpc_addr_t a )                               { o->jsr( a ); }
void nrpc_send_block( nrpc_t* o, const byte* p, int s )                 { o->send_block( p, s ); }
void nrpc_delay_cycles( nrpc_t* o, int n )                              { o->delay_cycles( n ); }
void nrpc_delay_bytes( nrpc_t* o, int n )                               { o->delay_bytes( n ); }
void nrpc_delay_msec( nrpc_t* o, int n )                                { o->delay_msec( n ); }

nrpc_err_t nrpc_load_mem_library( nrpc_t* o, const byte* p, int s )     { return o->load_library( p, s ); }
const unsigned char* nrpc_recording( nrpc_t* o, int* size_out )         { return o->recording( size_out ); }
int nrpc_57600_count( const nrpc_t* o )                                 { return o->recording_57600_count(); }
void nrpc_clear_recording( nrpc_t* o )                                  { o->clear_recording(); }
void nrpc_send( nrpc_t* o, const byte* p, int s )                       { o->send( p, s ); }
void nrpc_send_raw( nrpc_t* o, const byte* p, int s )                   { o->send_raw( p, s ); }

void nrpc_zero_pad( nrpc_t* o, int count )
{
	byte zero [1] = { 0 };
	
	while ( count-- )
		o->send_raw( zero, sizeof zero );
	
	o->delay_bytes( 15 );
}

nrpc_t* nrpc_new( int flags )
{
	nrpc_t* p = new nrpc_t;
	if ( p != NULL )
	{
		// Ignore errors, since we haven't loaded library yet
		if ( flags & nrpc_pal && p->enable_pal_serial() )
			{ }
	
		if ( flags & nrpc_115200 && p->enable_115200_serial() )
			{ }
	}
	
	return p;
}

void nrpc_call( nrpc_t* o, const char name [],
		int arg0, int arg1, int arg2, int arg3 )
{
	o->call( name, arg0, arg1, arg2, arg3 );
}

void nrpc_call_extra( nrpc_t* o, const char name [], const byte* extra, int extra_size,
		int arg0, int arg1, int arg2, int arg3 )
{
	o->call_extra( name, extra, extra_size, arg0, arg1, arg2, arg3 );
}

void nrpc_call_code( nrpc_t* o, nrpc_addr_t a, const byte* p, int s,
		nrpc_addr_t exec, const byte* extra , int extra_size,
		int arg0, int arg1, int arg2, int arg3 )
{
	o->call_code( a, p, s, exec, extra, extra_size, arg0, arg1, arg2, arg3 );
}

static nrpc_err_t save_file( const char path [], const void* in, int size )
{
	FILE* out = fopen( path, "wb" );
	if ( out == NULL )
		return "couldn't create file";
	
	fwrite( in, 1, size, out );
	if ( fclose( out ) )
		return "couldn't write file";
	
	return NULL;
}

nrpc_err_t nrpc_save_split_recording( nrpc_t* o, const char path_57600 [],
		const char path_115200 [] )
{
	nrpc_err_t err;
	const unsigned char* data;
	int remain = 0;
	int size;
	
	if ( nrpc_missing_routines( o ) )
		return "routines are missing from library";
	
	data = nrpc_recording( o, &size );
	if ( data == NULL )
		return "out of memory";
	
	if ( path_115200 )
	{
		remain = size;
		size = o->recording_57600_count();
		remain -= size;
	}
	
	err = save_file( path_57600, data, size );
	
	if ( !err && path_115200 )
		err = save_file( path_115200, data + size, remain );
	
	return NULL;
}

nrpc_err_t nrpc_save_recording( nrpc_t* o, const char path [] )
{
	return nrpc_save_split_recording( o, path, NULL );
}
