/** NES remote procedure call recording interface \file */

#ifndef BLARGG_NRPC_H
#define BLARGG_NRPC_H

#ifdef __cplusplus
	extern "C" {
#endif

/** First parameter of most functions is a pointer to nrpc_t. */
typedef struct nrpc_t nrpc_t;

/** NES CPU or PPU address */
typedef int nrpc_addr_t;

/** Boolean, 1 if true */
typedef int nrpc_bool_t;

	#ifndef nrpc_err_t /* allows easier testing */

/** Pointer to error string, or NULL if function was successful */
typedef const char* nrpc_err_t;

	#endif


/**** Basics ****/

/** Returns pointer to a new recorder, or NULL if out of memory. Pass
0 for NTSC 57600, nrpc_pal for PAL 57600, and OR with nrpc_115200 for
115200 serial. */
nrpc_t* nrpc_new( int flags );
enum { nrpc_ntsc   = 0x00 };
enum { nrpc_pal    = 0x01 };
enum { nrpc_57600  = 0x00 };
enum { nrpc_115200 = 0x02 };

/** Deletes recording, freeing memory. OK to pass NULL. */
void nrpc_delete( nrpc_t* );

/** Loads library of routines from file. Previously loaded routines
are retained, with new ones of the same name replacing them. Libraries
can be loaded at any time. */
nrpc_err_t nrpc_load_library( nrpc_t*, const char path [] );

/** Saves recording to file, ready to send to NES */
nrpc_err_t nrpc_save_recording( nrpc_t*, const char path [] );


/**** Writing ****/

/** Writes byte in CPU address space */
void nrpc_write_byte( nrpc_t*, nrpc_addr_t, int value );

/** Writes buffer in CPU address space. */
void nrpc_write_mem( nrpc_t*, nrpc_addr_t, const unsigned char* in, int size );

/** Writes value bytes in CPU address space */
void nrpc_fill_mem( nrpc_t*, nrpc_addr_t, int value, int size );

/** Writes buffer in PPU address space */
void nrpc_write_ppu( nrpc_t*, nrpc_addr_t, const unsigned char* in, int size );

/** Writes value to bytes in PPU address space */
void nrpc_fill_ppu( nrpc_t*, nrpc_addr_t, int value, int size );

/** Writes to MMC1. If reset is true, resets shift register first. */
void nrpc_write_mmc1( nrpc_t*, nrpc_addr_t, int value, nrpc_bool_t reset );


/**** Reading ****/

/* Data is sent back to PC, without any protocol. Use nrpc_read_crc() along
with nrpc_calc_crc() to verify integrity of received data. */

/** Reads one or more bytes from CPU address space */
void nrpc_read_mem( nrpc_t*, nrpc_addr_t, int size );

/** Reads one or more bytes from PPU address space */
void nrpc_read_ppu( nrpc_t*, nrpc_addr_t, int size );

/** Reads 2-byte CRC of all data before this, then resets it.
Sent MSB first, allowing it to be CRC'd along with the data and get 0
if the CRC was correct. */
void nrpc_read_crc( nrpc_t* );

/** Calculates CRC of received data. After all data and 2-byte CRC have
been passed, returns nrpc_correct_crc if correct. Pass 0 for old_crc
if all data and CRC are in one block, otherwise previous value. */
int nrpc_calc_crc( nrpc_t*, const unsigned char* in, int count, int old_crc );
enum { nrpc_crc_size = 2 };
enum { nrpc_correct_crc = 0x99 };


/**** User-defined ****/

/** Calls routine already in memory at address */
void nrpc_jsr( nrpc_t*, nrpc_addr_t );

/** Calls named routine with arguments. Pass 0 for unused arguments. */
void nrpc_call( nrpc_t*, const char name [],
		int arg0, int arg1, int arg2, int arg3 );

/** Same as npc_call_routine(), but also appends extra_size bytes of data
to end of routine. */
void nrpc_call_extra( nrpc_t*, const char name [],
		const unsigned char* extra, int extra_size,
		int arg0, int arg1, int arg2, int arg3 );

/** Sends checksummed block of data to routine just called. The routine
should call begin_block_read, read, and end_block_read to receive it.
Any block size is supported, including greater than 64KB. */
void nrpc_send_block( nrpc_t*, const unsigned char* in, int size );

/** Delays allow routine to take more time before exiting back to loader. */

/** Delays by specified number of cycles before calling next routine. */
void nrpc_delay_cycles( nrpc_t*, int cycles );

/** Delays by specified number of serial bytes before calling next routine. */
void nrpc_delay_bytes( nrpc_t*, int bytes );

/** Delays by specified number of milliseconds. */
void nrpc_delay_msec( nrpc_t*, int msec );


/**** Advanced ****/

/** Inserts specified number of zero bytes into output, followed by several 0xFF
bytes. Meant for use with auto bootloader. */
void nrpc_zero_pad( nrpc_t*, int count );

/** Same as nrpc_load_library(), but loads from memory. Makes copy of data. */
nrpc_err_t nrpc_load_mem_library( nrpc_t*, const unsigned char* data, int size );

/** Returns pointer and size of recording data to send to NES. Does not clear it.
NULL if there was an error. */
const unsigned char* nrpc_recording( nrpc_t*, int* size_out );

/** Number of bytes at beginning of recording that must be sent at 57600 */
int nrpc_57600_count( const nrpc_t* );

/** Saves 57600 and 115200 portions of recording separately. If 115200 isn't
even enabled, writes zero-length file at path_57600. */
nrpc_err_t nrpc_save_split_recording( nrpc_t*, const char path_57600 [],
		const char path_115200 [] );

/** Clears recorded data */
void nrpc_clear_recording( nrpc_t* );

/** True if named routine exists in library */
nrpc_bool_t nrpc_routine_exists( nrpc_t*, const char name [] );

/** True if any new attempts were made to call routines not in library.
Clears flag on return. Automatically checked by nrpc_save_recording(). */
nrpc_bool_t nrpc_missing_routines( nrpc_t* );

/** Uploads size bytes of code to dest in NES memory, then executes it at exec
Appends extra_size bytes to routine's code. Passes 16-bit arguments to routine. */
void nrpc_call_code( nrpc_t*, nrpc_addr_t dest, const unsigned char* in, int size,
		nrpc_addr_t exec, const unsigned char* extra , int extra_size,
		int arg0, int arg1, int arg2, int arg3 );

/** Sends individual bytes without any additional protocol */
void nrpc_send( nrpc_t*, const unsigned char* in, int size );

/** Writes bytes without any encoding. Won't be readable by the loader's
standard read routine. Meant for custom receive routine which doesn't use
scrambled byte format. */
void nrpc_send_raw( nrpc_t*, const unsigned char* in, int size );

#ifdef __cplusplus
	}
#endif

#endif
